/**
  ******************************************************************************
  * @file i2c_ee.c
  * @brief This file provides a set of functions needed to manage the 
	* communication between I2C peripheral and I2C M24C64 EEPROM.
  * @author STMicroelectronics - MCD Application Team
  * @version V1.0.0
  * @date 04/10/2009
  ******************************************************************************
  *
  * THE PRESENT FIRMWARE WHICH IS FOR GUIDANCE ONLY AIMS AT PROVIDING CUSTOMERS
  * WITH CODING INFORMATION REGARDING THEIR PRODUCTS IN ORDER FOR THEM TO SAVE
  * TIME. AS A RESULT, STMICROELECTRONICS SHALL NOT BE HELD LIABLE FOR ANY
  * DIRECT, INDIRECT OR CONSEQUENTIAL DAMAGES WITH RESPECT TO ANY CLAIMS ARISING
  * FROM THE CONTENT OF SUCH FIRMWARE AND/OR THE USE MADE BY CUSTOMERS OF THE
  * CODING INFORMATION CONTAINED HEREIN IN CONNECTION WITH THEIR PRODUCTS.
  *
  * <h2><center>&copy; COPYRIGHT 2009 STMicroelectronics</center></h2>
  * @image html logo.bmp
  ******************************************************************************
  */
/* Includes ------------------------------------------------------------------*/
#include "i2c_ee.h"
#include "stm8l10x_i2c.h"

/* Private typedef -----------------------------------------------------------*/
/* Private define ------------------------------------------------------------*/
#define I2C_Speed              100000
#define I2C1_SLAVE_ADDRESS7    0xA0
#define I2C2_SLAVE_ADDRESS7    0x90
#define EEPROM_BASE_ADDRESS    0x0000
/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
uint8_t EEPROM_ADDRESS = 0xA0;
uint8_t SENSOR_ADDRESS = 0x90;
/* Private function prototypes -----------------------------------------------*/
ErrorStatus v = 0;
FlagStatus  f = 0;
unsigned int cpt = 0 ;
unsigned int timeout = 60000;
/* Private functions ---------------------------------------------------------*/

/**
  * @brief Initializes the I2C peripheral registers to communicate the EEPROM
	* and the temperature SENSOR
  * @par Parameters:
  * None
  * @retval None
  * @par Required preconditions:
  * None
  */

void I2C_EEInit_EEPROM(void)
{
	I2C_DeInit();
  /* I2C Peripheral Enable */
  I2C_Cmd(ENABLE);
  /* Apply I2C configuration after enabling it on EEPROM  */
	I2C_Init(I2C_Speed, I2C1_SLAVE_ADDRESS7, I2C_DutyCycle_2, I2C_Ack_Enable, I2C_AcknowledgedAddress_7bit);

	
}

void I2C_EEInit_SENSOR(void)
{
  /* I2C Peripheral Enable */
  I2C_Cmd(ENABLE);
	/* Apply I2C configuration after enabling it on EEPROM & SENSOR */
	/* EEPROM */
	I2C_Init(I2C_Speed, I2C1_SLAVE_ADDRESS7, I2C_DutyCycle_2, I2C_Ack_Enable, I2C_AcknowledgedAddress_7bit);
	/* SENSOR */
	I2C_Init(I2C_Speed, I2C2_SLAVE_ADDRESS7, I2C_DutyCycle_2, I2C_Ack_Enable, I2C_AcknowledgedAddress_7bit);
	
}

/**
  * @brief Writes one byte to the I2C EEPROM.
  * @param[in] pBuffer Pointer to the buffer containing the data to be 
	* written to the EEPROM.
  * @param[in] WriteAddr EEPROM's internal address to write to.
  * @retval None
  * @par Required preconditions:
  * None
  */
ErrorStatus I2C_EE_ByteWrite(uint8_t* pBuffer, uint16_t WriteAddr)
{
  /* Send STRAT condition */
  I2C_GenerateSTART(ENABLE);

  /* Test on EV5 and clear it I2C_EVENT_MASTER_MODE_SELECT */
	v = ERROR;
	cpt = 0;
	while(cpt<timeout && v == ERROR)
	{
		v = I2C_CheckEvent(I2C_EVENT_MASTER_MODE_SELECT);
		cpt ++;
	}
	if(v != ERROR)
	{
		/* Send EEPROM address for write */
		I2C_Send7bitAddress(EEPROM_ADDRESS, I2C_Direction_Transmitter);
		
		/* Test on EV6 and clear it */
		v = ERROR;
		cpt = 0;
		while(cpt<timeout && v == ERROR)
		{
			I2C_CheckEvent(I2C_EVENT_MASTER_TRANSMITTER_MODE_SELECTED);
			cpt ++;
		}
		if(v != ERROR)
		{
			I2C_GetFlagStatus(I2C_FLAG_ADDR);
			(void)(I2C->SR3);
			
			/* Send Address (on 2 bytes) of first byte to be written & wait event detection */
			I2C_SendData((uint8_t)(WriteAddr >> 8)); /* MSB */
			/* Test on EV8 and clear it */
			v = ERROR;
			cpt = 0;
			while(cpt<timeout && v == ERROR)
			{
				v = I2C_CheckEvent(I2C_EVENT_MASTER_BYTE_TRANSMITTING);
				cpt ++;
			}
			if(v != ERROR)
			{
				I2C_SendData((uint8_t)(WriteAddr)); /* LSB */
				/* Test on EV8 and clear it */
				v = ERROR;
				cpt = 0;
				while(cpt<timeout && v == ERROR)
				{
					v = I2C_CheckEvent(I2C_EVENT_MASTER_BYTE_TRANSMITTING);
					cpt ++;
				}
				if(v != ERROR)
				{
					/* Send the byte to be written */
					I2C_SendData(*pBuffer); 
					
					/* Test on EV8 and clear it */
					v = ERROR;
					cpt = 0;
					while(cpt<timeout && v == ERROR)
					{
						v = I2C_CheckEvent(I2C_EVENT_MASTER_BYTE_TRANSMITTED);
						cpt ++;
					}
					if(v != ERROR)
					{
						/* Send STOP condition */
						//I2C_GenerateSTOP(ENABLE);
						v = SUCCESS;
					}
				}
			}
		}
	}
	I2C_GenerateSTOP(ENABLE);
	return v;
}

/**
  * @brief Writes two bytes to the I2C SENSOR.
  * @param[in] ConfigBytes unit 16 containing the data to be 
	* written to the SENSOR for configure it.
  * @retval None
  * @par Required preconditions:
  * None
  */
ErrorStatus I2C_SS_Config(uint16_t ConfigBytes)
{
  I2C_GenerateSTART(ENABLE);

  /* Test on EV5 and clear it I2C_EVENT_MASTER_MODE_SELECT */
	v = ERROR;
	cpt = 0;
	while(cpt<timeout && v == ERROR)
	{
		v = I2C_CheckEvent(I2C_EVENT_MASTER_MODE_SELECT);
		cpt ++;
	}
	if(v != ERROR)
	{
		/* Send SENSOR 7bit address for write */
		I2C_Send7bitAddress(SENSOR_ADDRESS, I2C_Direction_Transmitter);
		
		/* Test on EV6 and clear it */
		v = ERROR;
		cpt = 0;
		while(cpt<timeout && v == ERROR)
		{
			v = I2C_CheckEvent(I2C_EVENT_MASTER_TRANSMITTER_MODE_SELECTED);
			cpt ++;
		}
		if(v != ERROR)
		{
			I2C_GetFlagStatus(I2C_FLAG_ADDR);
			(void)(I2C->SR3);
			
			/* Send Pointer (on 1 byte) first byte to be written & wait event detection */
			I2C_SendData((uint8_t)(ConfigBytes >> 8)); /* MSB (pointer Byte) */
			/* Test on EV8 and clear it */
			v = ERROR;
			cpt = 0;
			while(cpt<timeout && v == ERROR)
			{
				v = I2C_CheckEvent(I2C_EVENT_MASTER_BYTE_TRANSMITTING);
				cpt ++;
			}
			if(v != ERROR)
			{
				/* Send config (on 1 byte) first byte to be written & wait event detection */
				I2C_SendData((uint8_t)(ConfigBytes)); /* LSB (config Byte)*/
				/* Test on EV8 and clear it */
				v = ERROR;
				cpt = 0;
				while(cpt<timeout && v == ERROR)
				{
					v = I2C_CheckEvent(I2C_EVENT_MASTER_BYTE_TRANSMITTING);
					cpt ++;
				}
				if(v != ERROR)
				{
					/* Send STOP condition */
					//I2C_GenerateSTOP(ENABLE);
					v = SUCCESS;
				}
			}
		}
	}
	I2C_GenerateSTOP(ENABLE);
	return v;
}


/**
  * @brief Writes more than one byte to the EEPROM with a single WRITE cycle. 
  *  The number of byte can't exceed the EEPROM page size.
  * @param[in] pBuffer Pointer to the buffer containing the data to be 
	* written to the EEPROM.
  * @param[in] WriteAddr EEPROM's internal address to write to.
  * @param[in] NumByteToWrite EEPROM's number of bytes to write to the EEPROM.  
  * @retval None
  * @par Required preconditions:
  * None
  */
ErrorStatus I2C_EE_PageWrite(uint8_t* pBuffer, uint16_t WriteAddr, uint8_t NumByteToWrite)
{
  /* While the bus is busy */
	f = SET;
	cpt = 0;
	
  while(cpt != timeout && f == SET)
  {
		f = I2C_GetFlagStatus(I2C_FLAG_BUSY);
		cpt ++;
		
  }

  if(f == RESET)
  {
	  I2C_GenerateSTART(ENABLE);
  
	  /* Test on EV5 and clear it */
	  v = ERROR;
	  cpt = 0;
	  while(cpt<timeout && v == ERROR)
	  {
			v = I2C_CheckEvent(I2C_EVENT_MASTER_MODE_SELECT);
			cpt ++;
	  }
	  if(v != ERROR)
	  {
		  /* Send EEPROM address for write */
		  I2C_Send7bitAddress(EEPROM_ADDRESS, I2C_Direction_Transmitter);

		  /* Test on EV6 and clear it */
			v = ERROR;
			cpt = 0;
			while(cpt<timeout && v == ERROR)
			{
				v = I2C_CheckEvent(I2C_EVENT_MASTER_TRANSMITTER_MODE_SELECTED);
				cpt ++;
			}
			
			if(v != ERROR)
			{
				I2C_GetFlagStatus(I2C_FLAG_ADDR);
				(void)(I2C->SR3);
	
				/* Send Address (on 2 bytes) of first byte to be written & wait event detection */
				I2C_SendData((uint8_t)(WriteAddr >> 8)); /* MSB */
				/* Test on EV8 and clear it */
				v = ERROR;
				cpt = 0;
				while(cpt<timeout && v == ERROR)
				{
					v = I2C_CheckEvent(I2C_EVENT_MASTER_BYTE_TRANSMITTING);
					cpt ++;
				}
				if(v != ERROR)
				{
					I2C_SendData((uint8_t)(WriteAddr)); /* LSB */
					/* Test on EV8 and clear it */
					v = ERROR;
					cpt = 0;
					while(cpt<timeout && v == ERROR)
					{
						v = I2C_CheckEvent(I2C_EVENT_MASTER_BYTE_TRANSMITTING);
						cpt ++;
					}
					if(v != ERROR)
					{
						/* While there is data to be written */
						while(NumByteToWrite--)  
						{
							/* Send the current byte */
							I2C_SendData(*pBuffer); 
				
							/* Point to the next byte to be written */
							pBuffer++; 
					
							/* Test on EV8 and clear it */
							while (!I2C_CheckEvent(I2C_EVENT_MASTER_BYTE_TRANSMITTED));
						}
			
						/* Send STOP condition */
						//I2C_GenerateSTOP(ENABLE);
						v = SUCCESS;
					}
				}
			}
	  }
  }
	I2C_GenerateSTOP(ENABLE);
	return v;
}

/**
  * @brief Reads a block of data from the EEPROM. 
  * @param[in] pBuffer pointer to the buffer that receives the data read
  * from the EEPROM.
  * @param[in] WriteAddr EEPROM's internal address to read from.
  * @param[in] NumByteToWrite EEPROM's number of bytes to read from the EEPROM.  
  * @retval None
  * @par Required preconditions:
  * None
  */
ErrorStatus I2C_EE_BufferRead(uint8_t* pBuffer, uint16_t ReadAddr, uint8_t NumByteToRead)
{ 
	/* While the bus is busy */
	f = SET;
	cpt = 0;
  while(cpt<timeout && f == SET)
  {
		f = I2C_GetFlagStatus(I2C_FLAG_BUSY);
		cpt ++;
  }

  if (cpt==timeout)
		cpt++;
		
  if(f == RESET)/* if the bus is not BUSY */
  {
		/* Generate start & wait event detection */
		I2C_GenerateSTART(ENABLE);
		
		/* Test on EV5 and clear it */
		v = ERROR;
		cpt = 0;
		while(cpt<timeout && v == ERROR)
		{
			v = I2C_CheckEvent(I2C_EVENT_MASTER_MODE_SELECT);
			cpt ++;
		}
		if(v != ERROR)
		{
			/* Send slave Address in write direction & wait detection event */
			I2C_Send7bitAddress(EEPROM_ADDRESS, I2C_Direction_Transmitter);
			/* Test on EV6 and clear it */
			v = ERROR;
			cpt = 0;
			while(cpt<timeout && v == ERROR)
			{
				v = I2C_CheckEvent(I2C_EVENT_MASTER_TRANSMITTER_MODE_SELECTED);
				cpt ++;
			}
			if( v!= ERROR)
			{
				I2C_GetFlagStatus(I2C_FLAG_ADDR);
				(void)(I2C->SR3);
					
				/* Send Address of first byte to be read & wait event detection */
				I2C_SendData((uint8_t)(ReadAddr >> 8)); /* MSB */
				/* Test on EV8 and clear it */
				v = ERROR;
				cpt = 0;
				while(cpt<timeout && v == ERROR)
				{
					v = I2C_CheckEvent(I2C_EVENT_MASTER_BYTE_TRANSMITTED);
					cpt ++;
				}
				if( v!= ERROR)
				{
					I2C_SendData((uint8_t)(ReadAddr)); /* LSB */
					/* Test on EV8 and clear it */
					v = ERROR;
					cpt = 0;
					while(cpt<timeout && v == ERROR)
					{
						v = I2C_CheckEvent(I2C_EVENT_MASTER_BYTE_TRANSMITTED);
						cpt ++;
					}
					if( v!= ERROR)
					{
						/* Send STRAT condition a second time */  
						I2C_GenerateSTART(ENABLE);
						/* Test on EV5 and clear it */
						v = ERROR;
						cpt = 0;
						while(cpt<timeout && v == ERROR)
						{
							v = I2C_CheckEvent(I2C_EVENT_MASTER_MODE_SELECT);
							cpt ++;
						}
						if( v!= ERROR)
						{
							/* Send slave Address in read direction & wait event */
							I2C_Send7bitAddress(EEPROM_ADDRESS, I2C_Direction_Receiver);
							/* Test on EV6 and clear it */
							v = ERROR;
							cpt = 0;
							while(cpt<timeout && v == ERROR)
							{
								v = I2C_CheckEvent(I2C_EVENT_MASTER_RECEIVER_MODE_SELECTED);
								cpt ++;
							}
							if( v!= ERROR)
							{
								I2C_GetFlagStatus(I2C_FLAG_ADDR);
								(void)(I2C->SR3);
							
								/* While there is data to be read */
								while(NumByteToRead)  
								{
									if(NumByteToRead == 1)
									{
										/* Disable Acknowledgement */
										I2C_AcknowledgeConfig(DISABLE);			
									}
							
									/* Test on EV7 and clear it */
									if(I2C_CheckEvent(I2C_EVENT_MASTER_BYTE_RECEIVED))
									{      
										/* Read a byte from the EEPROM */
										*pBuffer = I2C_ReceiveData();
							
										/* Point to the next location where the byte read will be saved */
										pBuffer++; 
										
										/* Decrement the read bytes counter */
										NumByteToRead--;    
									}
									if(NumByteToRead == 0)
									{
										/* Send STOP Condition */
										I2C_GenerateSTOP(ENABLE);
									}
								}
							
								/* Enable Acknowledgement to be ready for another reception */
								I2C_AckPositionConfig(I2C_AckPosition_Current);
								v = SUCCESS;
							}
							else
							{
								I2C_GenerateSTOP(ENABLE);
							}
						}
						else
						{
							I2C_GenerateSTOP(ENABLE);
						}
					}
					else
					{
						I2C_GenerateSTOP(ENABLE);
					}
				}
				else
				{
					I2C_GenerateSTOP(ENABLE);
				}
			}
			else
			{
				I2C_GenerateSTOP(ENABLE);
			}
		}
		else
		{
			I2C_GenerateSTOP(ENABLE);
		}
	}
	else
	{
		I2C_GenerateSTOP(ENABLE);
	}
	return v;
}


/**
  * @brief Reads 2 bytes temperature data from the SENSOR. 
  * @param[in] pBuffer pointer to the buffer that receives the data read
  * from the SENSOR.
  * @param[in] WriteAddr EEPROM's internal address to read from.
  * @param[in] NumByteToWrite EEPROM's number of bytes to read from the EEPROM.  
  * @retval None
  * @par Required preconditions:
  * None
  */
ErrorStatus I2C_SS_BufferRead(uint8_t* pBuffer, uint8_t Pointer_Byte, uint8_t NumByteToRead)
{  
	/* While the bus is busy */
	f = SET;
	cpt = 0;
  while(cpt<timeout && f == SET)
  {
		f = I2C_GetFlagStatus(I2C_FLAG_BUSY);
		cpt ++;
  }
  if(f== RESET)/* if the bus is not BUSY */
	{
		I2C_GenerateSTART(ENABLE);
		
		/* Test on EV5 and clear it */
		v = ERROR;
		cpt = 0;
		while(cpt<timeout && v == ERROR)
		{
			v = I2C_CheckEvent(I2C_EVENT_MASTER_MODE_SELECT);
			cpt ++;
		}
		if( v != ERROR)/* if Check EV5 OK */
		{
			/* Send slave Address in write direction & wait detection event */
			I2C_Send7bitAddress(SENSOR_ADDRESS, I2C_Direction_Transmitter);
			
			/* Test on EV6 and clear it */
			v = ERROR;
			cpt = 0;
			while(cpt<timeout && v == ERROR)
			{
				v = I2C_CheckEvent(I2C_EVENT_MASTER_TRANSMITTER_MODE_SELECTED);
				cpt ++;
			}
			if(v != ERROR)
			{
				I2C_GetFlagStatus(I2C_FLAG_ADDR);
				(void)(I2C->SR3);
					
				/* Configure the sensor in read mode Pointer 0x00 */
				I2C_SendData(Pointer_Byte); /* MSB */
				/* Test on EV8 and clear it */
				v = ERROR;
				cpt = 0;
				while(cpt<timeout && v == ERROR)
				{
					v = I2C_CheckEvent(I2C_EVENT_MASTER_BYTE_TRANSMITTED);
					cpt ++;
				}
				if(v != ERROR)
				{
					/* Send START condition a second time */  
					I2C_GenerateSTART(ENABLE);
					/* Test on EV5 and clear it */
					v = ERROR;
					cpt = 0;
					while(cpt<timeout && v == ERROR)
					{
						v = I2C_CheckEvent(I2C_EVENT_MASTER_MODE_SELECT);
						cpt ++;
					}
					if(v != ERROR)
					{
						/* Send slave Address in read direction & wait event */
						I2C_Send7bitAddress(SENSOR_ADDRESS, I2C_Direction_Receiver);
						/* Test on EV6 and clear it */
						while(cpt<timeout && v == ERROR)
						{
							v = I2C_CheckEvent(I2C_EVENT_MASTER_RECEIVER_MODE_SELECTED);
							cpt ++;
						}
						if(v != ERROR)
						{
							I2C_GetFlagStatus(I2C_FLAG_ADDR);
							(void)(I2C->SR3);
						
							/* While there is data to be read */
							while(NumByteToRead)  
							{
								if(NumByteToRead == 1)
								{
									/* Disable Acknowledgement */
									I2C_AcknowledgeConfig(DISABLE);			
								}
						
								/* Test on EV7 and clear it */
								if(I2C_CheckEvent(I2C_EVENT_MASTER_BYTE_RECEIVED))
								{      
									/* Read a byte from the SENSOR */
									*pBuffer = I2C_ReceiveData();
						
									/* Point to the next location where the byte read will be saved */
									pBuffer++; 
									
									/* Decrement the read bytes counter */
									NumByteToRead--;    
								}
								if(NumByteToRead == 0)
								{
									/* Send STOP Condition */
									I2C_GenerateSTOP(ENABLE);
								}
							}
							/* Enable Acknowledgement to be ready for another reception */
							I2C_AckPositionConfig(I2C_AckPosition_Current);
							v = SUCCESS;
						}
						else
						{
							I2C_GenerateSTOP(ENABLE);
						}
					}
					else
					{
						I2C_GenerateSTOP(ENABLE);
					}
				}
				else
				{
					I2C_GenerateSTOP(ENABLE);
				}
			}
			else
			{
				I2C_GenerateSTOP(ENABLE);
			}
		}
		else
		{
			I2C_GenerateSTOP(ENABLE);
		}
	}
	else
	{
		I2C_GenerateSTOP(ENABLE);
	}
	return v;
}

/******************* (C) COPYRIGHT 2009 STMicroelectronics *****END OF FILE****/
